/**
 * Grayscale Theme Switcher JavaScript
 * Version: 1.0.0
 */

(function($) {
    'use strict';

    // Variables
    let currentTheme = 'normal';
    let currentIntensity = 100;
    let isMenuOpen = false;

    // Initialize
    $(document).ready(function() {
        init();
    });

    function init() {
        // Check if preferences should be cleared
        checkAndClearPreferences();
        
        // Check if force override is enabled
        if (+gts_ajax.force_override) {
            // Force override is enabled - always use default theme
            currentTheme = gts_ajax.default_theme || 'normal';
            currentIntensity = parseInt(gts_ajax.intensity) || 100;
            
            // Apply the forced theme
            if (currentTheme !== 'normal') {
                applyTheme(currentTheme, currentIntensity);
            } else {
                clearAllFilters();
            }
            
            // Don't allow user changes if force override is on
            if (!Number(gts_ajax.allow_user_toggle)) {
                // Hide the button completely
                $('#gts-theme-switcher').hide();
            }
        } else {
            // Normal operation - respect user preferences
            loadUserPreferences();
            
            // Check if user has saved preference
            const hasSavedPreference = localStorage.getItem('gts_theme') || getCookie('gts_user_theme');
            
            // Only apply default theme if user hasn't set their own preference
            if (!hasSavedPreference && gts_ajax.default_theme && gts_ajax.default_theme !== 'normal') {
                applyTheme(gts_ajax.default_theme, gts_ajax.intensity);
                currentTheme = gts_ajax.default_theme;
                currentIntensity = parseInt(gts_ajax.intensity) || 100;
            } else if (hasSavedPreference && currentTheme === 'normal') {
                // Make sure normal theme is properly applied (no filters)
                clearAllFilters();
            }
        }
        
        // Setup event handlers
        setupEventHandlers();
        
        // Update UI
        updateUI();
        
        // Log debug info
        if (gts_ajax.debug) {
            console.log('GTS Init:', {
                force_override: gts_ajax.force_override,
                default_theme: gts_ajax.default_theme,
                current_theme: currentTheme,
                intensity: currentIntensity,
                user_can_toggle: gts_ajax.allow_user_toggle
            });
        }
    }
    
    function checkAndClearPreferences() {
        // Check if we need to clear preferences based on admin action
        const lastCleared = localStorage.getItem('gts_last_cleared') || 0;
        const clearTimestamp = gts_ajax.preferences_cleared || 0;
        const forceTimestamp = gts_ajax.force_override_timestamp || 0;
        
        // Clear if admin clicked clear button or just enabled force override
        if (clearTimestamp > lastCleared || (+gts_ajax.force_override && forceTimestamp > lastCleared)) {
            // Clear all stored preferences
            localStorage.removeItem('gts_theme');
            localStorage.removeItem('gts_intensity');
            
            // Clear cookies
            setCookie('gts_user_theme', '', -1);
            setCookie('gts_user_intensity', '', -1);
            
            // Update last cleared timestamp
            localStorage.setItem('gts_last_cleared', Math.max(clearTimestamp, forceTimestamp));
            
            if (gts_ajax.debug) {
                console.log('GTS: User preferences cleared by admin');
            }
        }
    }

    function loadUserPreferences() {
        // Try to get from localStorage first
        const savedTheme = localStorage.getItem('gts_theme');
        const savedIntensity = localStorage.getItem('gts_intensity');
        
        if (savedTheme) {
            currentTheme = savedTheme;
        } else if (getCookie('gts_user_theme')) {
            currentTheme = getCookie('gts_user_theme');
        } else {
            currentTheme = gts_ajax.default_theme || 'normal';
        }
        
        if (savedIntensity) {
            currentIntensity = parseInt(savedIntensity);
        } else if (getCookie('gts_user_intensity')) {
            currentIntensity = parseInt(getCookie('gts_user_intensity'));
        } else {
            currentIntensity = parseInt(gts_ajax.intensity) || 100;
        }
        
        // Apply saved preferences
        if (currentTheme !== 'normal') {
            applyTheme(currentTheme, currentIntensity);
        } else {
            // Make sure to clear any filters if theme is normal
            clearAllFilters();
        }
    }

    function setupEventHandlers() {
        // Toggle menu button
        $('#gts-toggle-btn').on('click', function(e) {
            e.stopPropagation();
            toggleMenu();
        });
        
        // Theme option buttons
        $('.gts-theme-option').on('click', function() {
            const theme = $(this).data('theme');
            selectTheme(theme);
        });
        
        // Intensity slider
        $('#gts-intensity-slider').on('input', function() {
            // Check if force override is enabled
            if (+gts_ajax.force_override) {
                // Reset to forced value
                $(this).val(gts_ajax.intensity);
                $('#gts-intensity-value').text(gts_ajax.intensity);
                return;
            }
            
            const value = $(this).val();
            $('#gts-intensity-value').text(value);
            currentIntensity = parseInt(value);
            
            // Apply intensity in real-time
            if (currentTheme !== 'normal') {
                applyTheme(currentTheme, currentIntensity);
            }
        });
        
        // Save intensity on change
        $('#gts-intensity-slider').on('change', function() {
            if (!Number(gts_ajax.force_override)) {
                saveUserPreferences();
            }
        });
        
        // Close menu when clicking outside
        $(document).on('click', function(e) {
            if (!$(e.target).closest('#gts-theme-switcher').length && isMenuOpen) {
                closeMenu();
            }
        });
        
        // Keyboard navigation
        $(document).on('keydown', function(e) {
            if (e.key === 'Escape' && isMenuOpen) {
                closeMenu();
            }
        });
        
        // Accessibility: Handle Enter and Space keys
        $('.gts-theme-option').on('keypress', function(e) {
            if (e.which === 13 || e.which === 32) {
                e.preventDefault();
                $(this).click();
            }
        });
    }

    function toggleMenu() {
        if (isMenuOpen) {
            closeMenu();
        } else {
            openMenu();
        }
    }

    function openMenu() {
        $('#gts-theme-menu').removeClass('gts-hidden');
        $('#gts-toggle-btn').attr('aria-expanded', 'true');
        isMenuOpen = true;
        
        // Update intensity slider value
        $('#gts-intensity-slider').val(currentIntensity);
        $('#gts-intensity-value').text(currentIntensity);
        
        // Focus first theme option for accessibility
        setTimeout(() => {
            $('.gts-theme-option').first().focus();
        }, 100);
    }

    function closeMenu() {
        $('#gts-theme-menu').addClass('gts-hidden');
        $('#gts-toggle-btn').attr('aria-expanded', 'false');
        isMenuOpen = false;
    }

    function selectTheme(theme) {
        // Check if force override is enabled
        if (+gts_ajax.force_override) {
            // Don't allow theme changes when force override is on
            alert('การเปลี่ยนธีมถูกปิดใช้งานโดยผู้ดูแลระบบ');
            return;
        }
        
        currentTheme = theme;
        
        // Remove all theme classes
        $('html').removeClass('gts-grayscale gts-sepia gts-dark gts-high-contrast');
        
        // Apply new theme or clear if normal
        if (theme !== 'normal') {
            applyTheme(theme, currentIntensity);
        } else {
            // Clear all filters when returning to normal
            clearAllFilters();
        }
        
        // Update UI
        updateUI();
        
        // Save preference
        saveUserPreferences();
        
        // Close menu after selection
        setTimeout(() => {
            closeMenu();
        }, 200);
    }

    function applyTheme(theme, intensity) {
        const $html = $('html');
        
        // Remove existing theme classes
        $html.removeClass('gts-grayscale gts-sepia gts-dark gts-high-contrast');
        
        // Clear existing filters first
        clearAllFilters();
        
        // Add new theme class and apply filters
        if (theme !== 'normal') {
            $html.addClass('gts-' + theme);
            
            // Apply intensity
            applyIntensity(theme, intensity);
        }
        
        // Trigger custom event
        $(document).trigger('gts:theme-changed', [theme, intensity]);
    }

    function clearAllFilters() {
        // Clear all CSS filters from html element
        $('html').css({
            'filter': '',
            '-webkit-filter': '',
            '-moz-filter': '',
            '-ms-filter': '',
            '-o-filter': ''
        });
        
        // Clear filters from media elements
        $('img, video, iframe, embed, object').css({
            'filter': '',
            '-webkit-filter': ''
        });
        
        // Remove all theme classes
        $('html').removeClass('gts-grayscale gts-sepia gts-dark gts-high-contrast');
    }

    function applyIntensity(theme, intensity) {
        const $html = $('html');
        let filterValue = '';
        
        switch(theme) {
            case 'grayscale':
                filterValue = `grayscale(${intensity}%)`;
                break;
            case 'sepia':
                filterValue = `sepia(${intensity}%)`;
                break;
            case 'dark':
                filterValue = `invert(${intensity}%) hue-rotate(180deg)`;
                break;
            case 'high-contrast':
                const contrastValue = 100 + (intensity / 2);
                filterValue = `contrast(${contrastValue}%)`;
                break;
        }
        
        // Apply filter with vendor prefixes
        $html.css({
            'filter': filterValue,
            '-webkit-filter': filterValue,
            '-moz-filter': filterValue,
            '-ms-filter': filterValue,
            '-o-filter': filterValue
        });
        
        // Special handling for dark mode images
        if (theme === 'dark') {
            $('img, video, iframe, embed, object').css({
                'filter': 'invert(100%) hue-rotate(180deg)',
                '-webkit-filter': 'invert(100%) hue-rotate(180deg)'
            });
        } else {
            $('img, video, iframe, embed, object').css({
                'filter': '',
                '-webkit-filter': ''
            });
        }
    }

    function updateUI() {
        // Update active state on theme options
        $('.gts-theme-option').removeClass('active');
        $(`.gts-theme-option[data-theme="${currentTheme}"]`).addClass('active');
        
        // Update button icon based on current theme
        updateButtonIcon();
    }

    function updateButtonIcon() {
        const $icon = $('#gts-toggle-btn .gts-icon');
        let icon = '🎨';
        
        switch(currentTheme) {
            case 'grayscale':
                icon = '⚫';
                break;
            case 'sepia':
                icon = '🟤';
                break;
            case 'dark':
                icon = '🌙';
                break;
            case 'high-contrast':
                icon = '⚡';
                break;
            default:
                icon = '🎨';
        }
        
        $icon.text(icon);
    }

    function saveUserPreferences() {
        // Don't save preferences if force override is enabled
        if (+gts_ajax.force_override) {
            if (gts_ajax.debug) {
                console.log('GTS: Not saving preferences - force override is enabled');
            }
            return;
        }
        
        // Save to localStorage
        localStorage.setItem('gts_theme', currentTheme);
        localStorage.setItem('gts_intensity', currentIntensity);
        
        // Also save to server via AJAX
        if (gts_ajax.ajax_url) {
            $.post(gts_ajax.ajax_url, {
                action: 'gts_save_preference',
                nonce: gts_ajax.nonce,
                theme: currentTheme,
                intensity: currentIntensity
            });
        }
        
        // Set cookies as fallback
        setCookie('gts_user_theme', currentTheme, 30);
        setCookie('gts_user_intensity', currentIntensity, 30);
    }

    // Cookie helper functions
    function setCookie(name, value, days) {
        const date = new Date();
        date.setTime(date.getTime() + (days * 24 * 60 * 60 * 1000));
        const expires = "expires=" + date.toUTCString();
        document.cookie = name + "=" + value + ";" + expires + ";path=/";
    }

    function getCookie(name) {
        const nameEQ = name + "=";
        const ca = document.cookie.split(';');
        for(let i = 0; i < ca.length; i++) {
            let c = ca[i];
            while (c.charAt(0) === ' ') c = c.substring(1, c.length);
            if (c.indexOf(nameEQ) === 0) return c.substring(nameEQ.length, c.length);
        }
        return null;
    }

    // Public API
    window.GTS = {
        setTheme: function(theme, intensity) {
            selectTheme(theme);
            if (intensity !== undefined) {
                currentIntensity = intensity;
                $('#gts-intensity-slider').val(intensity);
                $('#gts-intensity-value').text(intensity);
                applyTheme(currentTheme, currentIntensity);
                saveUserPreferences();
            }
        },
        getTheme: function() {
            return currentTheme;
        },
        getIntensity: function() {
            return currentIntensity;
        },
        reset: function() {
            this.setTheme('normal', 100);
        }
    };

    // Performance optimization: Debounce window resize
    let resizeTimer;
    $(window).on('resize', function() {
        clearTimeout(resizeTimer);
        resizeTimer = setTimeout(function() {
            // Reposition menu if needed
            if (isMenuOpen) {
                const $menu = $('#gts-theme-menu');
                const menuRect = $menu[0].getBoundingClientRect();
                
                // Check if menu is outside viewport
                if (menuRect.right > window.innerWidth) {
                    $menu.css('right', '10px');
                    $menu.css('left', 'auto');
                } else if (menuRect.left < 0) {
                    $menu.css('left', '10px');
                    $menu.css('right', 'auto');
                }
            }
        }, 250);
    });

    // Add smooth transitions when theme changes
    $(document).on('gts:theme-changed', function(e, theme, intensity) {
        // Add transition class
        $('body').addClass('gts-transitioning');
        
        setTimeout(function() {
            $('body').removeClass('gts-transitioning');
        }, 500);
        
        // Log for debugging (remove in production)
        console.log('Theme changed to:', theme, 'with intensity:', intensity);
    });

    // Handle prefers-color-scheme changes
    if (window.matchMedia) {
        const darkModeQuery = window.matchMedia('(prefers-color-scheme: dark)');
        darkModeQuery.addListener(function(e) {
            // Optional: Auto-switch to dark theme when system dark mode is enabled
            // if (e.matches && currentTheme === 'normal') {
            //     selectTheme('dark');
            // }
        });
    }

})(jQuery);

// Vanilla JS fallback if jQuery is not available
if (typeof jQuery === 'undefined') {
    console.warn('Grayscale Theme Switcher: jQuery not found. Running in vanilla JS mode.');
    
    document.addEventListener('DOMContentLoaded', function() {
        // Simplified vanilla JS implementation
        const toggleBtn = document.getElementById('gts-toggle-btn');
        const themeMenu = document.getElementById('gts-theme-menu');
        const themeOptions = document.querySelectorAll('.gts-theme-option');
        const intensitySlider = document.getElementById('gts-intensity-slider');
        const intensityValue = document.getElementById('gts-intensity-value');
        
        let currentTheme = localStorage.getItem('gts_theme') || 'normal';
        let currentIntensity = parseInt(localStorage.getItem('gts_intensity')) || 100;
        
        // Apply saved theme
        if (currentTheme !== 'normal') {
            document.documentElement.className = 'gts-' + currentTheme;
        }
        
        // Toggle menu
        if (toggleBtn) {
            toggleBtn.addEventListener('click', function(e) {
                e.stopPropagation();
                themeMenu.classList.toggle('gts-hidden');
            });
        }
        
        // Theme selection
        themeOptions.forEach(function(option) {
            option.addEventListener('click', function() {
                const theme = this.getAttribute('data-theme');
                
                // Remove all theme classes and filters
                document.documentElement.className = '';
                document.documentElement.style.filter = '';
                document.documentElement.style.webkitFilter = '';
                
                // Clear filters from media elements
                const mediaElements = document.querySelectorAll('img, video, iframe, embed, object');
                mediaElements.forEach(function(el) {
                    el.style.filter = '';
                    el.style.webkitFilter = '';
                });
                
                // Apply new theme if not normal
                if (theme !== 'normal') {
                    document.documentElement.className = 'gts-' + theme;
                    const filterValue = getFilterValue(theme, currentIntensity);
                    document.documentElement.style.filter = filterValue;
                    document.documentElement.style.webkitFilter = filterValue;
                }
                
                // Save preference
                localStorage.setItem('gts_theme', theme);
                currentTheme = theme;
                
                // Update active state
                themeOptions.forEach(opt => opt.classList.remove('active'));
                this.classList.add('active');
                
                // Close menu
                setTimeout(() => {
                    themeMenu.classList.add('gts-hidden');
                }, 200);
            });
        });
        
        // Intensity control
        if (intensitySlider) {
            intensitySlider.value = currentIntensity;
            intensityValue.textContent = currentIntensity;
            
            intensitySlider.addEventListener('input', function() {
                const value = this.value;
                intensityValue.textContent = value;
                currentIntensity = parseInt(value);
                
                // Apply intensity
                const filterValue = getFilterValue(currentTheme, currentIntensity);
                document.documentElement.style.filter = filterValue;
            });
            
            intensitySlider.addEventListener('change', function() {
                localStorage.setItem('gts_intensity', currentIntensity);
            });
        }
        
        // Close menu when clicking outside
        document.addEventListener('click', function(e) {
            if (!e.target.closest('#gts-theme-switcher')) {
                themeMenu.classList.add('gts-hidden');
            }
        });
        
        function getFilterValue(theme, intensity) {
            switch(theme) {
                case 'grayscale':
                    return `grayscale(${intensity}%)`;
                case 'sepia':
                    return `sepia(${intensity}%)`;
                case 'dark':
                    return `invert(${intensity}%) hue-rotate(180deg)`;
                case 'high-contrast':
                    return `contrast(${100 + intensity/2}%)`;
                default:
                    return 'none';
            }
        }
    });
}
